/*
Half-Life MAP viewing utility.
Copyright (C) 2003  Ryan Samuel Gregg

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#pragma once
#include "stdafx.h"
#include "WorldObject.h"
#include "Face.h"
#include "Frustum.h"
#include "TextureManager.h"

__gc class CBrush : public CWorldObject
{
private:
	Color3uc Color;
	bool bCulled;
	BoundingBox Box;
	ArrayList *Faces;

public:
	CBrush(CConfig *Config) : CWorldObject(Config)
	{
		bCulled = false;

		Faces = new ArrayList();
	}

	void SetColor(unsigned char R, unsigned char G, unsigned char B)
	{
		Color.R = R;
		Color.G = G;
		Color.B = B;
	}

	void AddFace(CFace *Face)
	{
		Faces->Add(Face);
	}

	ArrayList *GetFaces()
	{
		return Faces;
	}

	void UpdateBoundingBoxes()
	{
		CFace *Face;
		Vertex3f *Vertices = static_cast<CFace*>(Faces->get_Item(0))->GetVertices();

		Box.vNegBound.X = Vertices[0].X;
		Box.vNegBound.Y = Vertices[0].Y;
		Box.vNegBound.Z = Vertices[0].Z;
		Box.vPosBound.X = Vertices[0].X;
		Box.vPosBound.Y = Vertices[0].Y;
		Box.vPosBound.Z = Vertices[0].Z;

		for(int i = 0; i < Faces->Count; i++)
		{
			Face = static_cast<CFace*>(Faces->get_Item(i));

			Vertices = Face->GetVertices();

			for(int j = 0; j < Face->GetVertexCount(); j++)
			{
				if(Vertices[j].X < Box.vNegBound.X)
					Box.vNegBound.X = Vertices[j].X;

				if(Vertices[j].Y < Box.vNegBound.Y)
					Box.vNegBound.Y = Vertices[j].Y;

				if(Vertices[j].Z < Box.vNegBound.Z)
					Box.vNegBound.Z = Vertices[j].Z;

				if(Vertices[j].X > Box.vPosBound.X)
					Box.vPosBound.X = Vertices[j].X;

				if(Vertices[j].Y > Box.vPosBound.Y)
					Box.vPosBound.Y = Vertices[j].Y;

				if(Vertices[j].Z > Box.vPosBound.Z)
					Box.vPosBound.Z = Vertices[j].Z;
			}
		}
	}

	BoundingBox GetBoundingBox()
	{
		return Box;
	}

	bool CullBrush(CFrustum *Frustum)
	{
		bCulled = !Frustum->CubeInFrustum(Box);

		if(bCulled)
			Frustum->IncCulled();

		return bCulled;
	}

	void UpdateTedCoords(CTextureManager *TextureManager)
	{
		for(int i = 0; i < Faces->Count; i++)
		{
			static_cast<CFace*>(Faces->get_Item(i))->UpdateTexCoords(TextureManager);
		}
	}

	void DrawBrushTextured()
	{
		if(bCulled)
			return;

		for(int i = 0; i < Faces->Count; i++)
		{
			static_cast<CFace*>(Faces->get_Item(i))->DrawFaceTextured();
		}
	}

	void DrawBrushSolid()
	{
		if(bCulled)
			return;

		glColor3ub(Color.R, Color.G, Color.B);

		for(int i = 0; i < Faces->Count; i++)
		{
			static_cast<CFace*>(Faces->get_Item(i))->DrawFaceSolid();
		}
	}

	void DrawBrushWireFrame()
	{
		if(bCulled)
			return;

		glColor3ub(Color.R, Color.G, Color.B);

		for(int i = 0; i < Faces->Count; i++)
		{
			static_cast<CFace*>(Faces->get_Item(i))->DrawFaceWireFrame();
		}
	}

	void DrawBrushPoints()
	{
		if(bCulled)
			return;

		glColor3ub(Color.R, Color.G, Color.B);

		for(int i = 0; i < Faces->Count; i++)
		{
			static_cast<CFace*>(Faces->get_Item(i))->DrawFacePoints();
		}
	}

	void Highlight(bool bPrepared)
	{
		if(!bPrepared)
		{
			PrepareHighlight();
		}

		for(int i = 0; i < Faces->Count; i++)
		{
			static_cast<CFace*>(Faces->get_Item(i))->Highlight(true);
		}
	}

	void Outline()
	{
		for(int i = 0; i < Faces->Count; i++)
		{
			static_cast<CFace*>(Faces->get_Item(i))->Outline();
		}
	}
};